-- Script para criar o banco de dados e tabelas básicas
CREATE DATABASE IF NOT EXISTS sistema_franqueados;
USE sistema_franqueados;

-- Tabela de usuários
CREATE TABLE IF NOT EXISTS usuarios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) NOT NULL UNIQUE,
    senha VARCHAR(255) NOT NULL,
    tipo ENUM('admin', 'gestor', 'franqueado') NOT NULL DEFAULT 'franqueado',
    status TINYINT(1) NOT NULL DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Inserir usuário administrador padrão (senha: admin123)
INSERT INTO usuarios (nome, email, senha, tipo) VALUES 
('Administrador', 'admin@sistema.com', '$2y$10$NmsIZRZ7r5O1Oa0WiQRiG.gGC0MppCkZQ0XCQ/eOZ7TZs4qqm68Oa', 'admin')
ON DUPLICATE KEY UPDATE nome = 'Administrador';

-- Tabela de franquias
CREATE TABLE IF NOT EXISTS franquias (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    cnpj VARCHAR(20),
    telefone VARCHAR(20),
    email VARCHAR(100),
    endereco TEXT,
    status TINYINT(1) NOT NULL DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tabela de relação entre usuários e franquias
CREATE TABLE IF NOT EXISTS usuario_franquia (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    franquia_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (franquia_id) REFERENCES franquias(id) ON DELETE CASCADE,
    UNIQUE KEY (usuario_id, franquia_id)
);

-- Tabela de arquivos
CREATE TABLE IF NOT EXISTS arquivos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(255) NOT NULL,
    caminho VARCHAR(255) NOT NULL,
    tipo VARCHAR(50) NOT NULL,
    tamanho BIGINT NOT NULL,
    categoria ENUM('documento', 'imagem', 'video') NOT NULL,
    visibilidade ENUM('privado', 'publico') NOT NULL DEFAULT 'privado',
    usuario_id INT,
    franquia_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE SET NULL,
    FOREIGN KEY (franquia_id) REFERENCES franquias(id) ON DELETE SET NULL
);

-- Tabela de pastas
CREATE TABLE IF NOT EXISTS pastas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    pasta_pai_id INT,
    usuario_id INT,
    franquia_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (pasta_pai_id) REFERENCES pastas(id) ON DELETE CASCADE,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (franquia_id) REFERENCES franquias(id) ON DELETE CASCADE
);

-- Tabela de relação entre arquivos e pastas
CREATE TABLE IF NOT EXISTS arquivo_pasta (
    id INT AUTO_INCREMENT PRIMARY KEY,
    arquivo_id INT NOT NULL,
    pasta_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (arquivo_id) REFERENCES arquivos(id) ON DELETE CASCADE,
    FOREIGN KEY (pasta_id) REFERENCES pastas(id) ON DELETE CASCADE,
    UNIQUE KEY (arquivo_id, pasta_id)
);

-- Tabela de notificações
CREATE TABLE IF NOT EXISTS notificacoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    titulo VARCHAR(255) NOT NULL,
    mensagem TEXT NOT NULL,
    tipo ENUM('info', 'success', 'warning', 'error') NOT NULL DEFAULT 'info',
    usuario_id INT,
    franquia_id INT,
    para_todos TINYINT(1) NOT NULL DEFAULT 0,
    lida TINYINT(1) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (franquia_id) REFERENCES franquias(id) ON DELETE CASCADE
);

-- Tabela de eventos
CREATE TABLE IF NOT EXISTS eventos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    titulo VARCHAR(255) NOT NULL,
    descricao TEXT,
    data_inicio DATETIME NOT NULL,
    data_fim DATETIME NOT NULL,
    local VARCHAR(255),
    usuario_id INT,
    franquia_id INT,
    para_todos TINYINT(1) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (franquia_id) REFERENCES franquias(id) ON DELETE CASCADE
); 