<?php
/**
 * Página de upload de arquivos
 */

$titulo_pagina = 'Upload de Arquivos';

// Obter informações do usuário
$user_id = $_SESSION['user_id'];
$user_type = $_SESSION['user_type'];

// Carregar modelos
$usuario = new Usuario();
$franquia = new Franquia();
$pasta = new Pasta();

// Obter franquias do usuário
$franquias_usuario = $usuario->getFranquias($user_id);

// Verificar pasta selecionada
$pasta_id = isset($_GET['pasta']) ? (int)$_GET['pasta'] : null;
$pasta_atual = null;

if ($pasta_id) {
    $pasta_atual = $pasta->getById($pasta_id);
    
    // Verificar se a pasta existe
    if (!$pasta_atual) {
        set_flash_message('danger', 'Pasta não encontrada.');
        redirect('/arquivos.php');
    }
    
    // Verificar permissão
    $permitido = false;
    
    if ($pasta_atual['usuario_id'] == $user_id) {
        // É o proprietário
        $permitido = true;
    } elseif ($pasta_atual['franquia_id'] && in_array($pasta_atual['franquia_id'], array_column($franquias_usuario, 'id'))) {
        // Pertence a uma franquia do usuário
        $permitido = true;
    } elseif (is_admin()) {
        // Administrador tem acesso a tudo
        $permitido = true;
    }
    
    if (!$permitido) {
        set_flash_message('danger', 'Você não tem permissão para fazer upload nesta pasta.');
        redirect('/arquivos.php');
    }
}

// Processar upload (método tradicional, não AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['arquivo'])) {
    $visibilidade = $_POST['visibilidade'] ?? 'privado';
    $franquia_id = !empty($_POST['franquia_id']) ? (int)$_POST['franquia_id'] : null;
    $pasta_id = !empty($_POST['pasta_id']) ? (int)$_POST['pasta_id'] : null;
    
    $arquivo = new Arquivo();
    $resultado = $arquivo->processarUpload($_FILES['arquivo'], [
        'usuario_id' => $user_id,
        'franquia_id' => $franquia_id,
        'pasta_id' => $pasta_id,
        'visibilidade' => $visibilidade
    ]);
    
    if ($resultado['sucesso']) {
        set_flash_message('success', 'Arquivo enviado com sucesso!');
        
        // Redirecionar para a pasta onde o arquivo foi enviado
        if ($pasta_id) {
            redirect('/arquivos.php?pasta=' . $pasta_id);
        } else {
            redirect('/arquivos.php');
        }
    } else {
        set_flash_message('danger', 'Erro ao enviar arquivo: ' . $resultado['mensagem']);
    }
}

// Iniciar buffer de saída para capturar o conteúdo
ob_start();
?>

<div class="container-fluid">
    <!-- Barra de navegação -->
    <nav aria-label="breadcrumb" class="mb-4">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="/arquivos.php">Meus Arquivos</a></li>
            <?php if ($pasta_atual): ?>
            <li class="breadcrumb-item"><a href="/arquivos.php?pasta=<?php echo $pasta_atual['id']; ?>"><?php echo htmlspecialchars($pasta_atual['nome']); ?></a></li>
            <?php endif; ?>
            <li class="breadcrumb-item active">Upload de Arquivos</li>
        </ol>
    </nav>
    
    <div class="row">
        <div class="col-lg-8 mx-auto">
            <div class="card shadow">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">
                        <i class="fas fa-upload me-2"></i> Upload de Arquivos
                    </h5>
                </div>
                <div class="card-body">
                    <!-- Mensagem de extensões suportadas -->
                    <div class="alert alert-info mb-4">
                        <h6 class="alert-heading mb-2"><i class="fas fa-info-circle me-2"></i> Informações importantes:</h6>
                        <ul class="mb-0">
                            <li>Tamanho máximo: <?php echo format_file_size(MAX_UPLOAD_SIZE); ?></li>
                            <li>
                                Extensões permitidas:
                                <ul>
                                    <li>Documentos: <?php echo implode(', ', ALLOWED_EXTENSIONS['documento']); ?></li>
                                    <li>Imagens: <?php echo implode(', ', ALLOWED_EXTENSIONS['imagem']); ?></li>
                                    <li>Vídeos: <?php echo implode(', ', ALLOWED_EXTENSIONS['video']); ?></li>
                                </ul>
                            </li>
                        </ul>
                    </div>
                    
                    <!-- Formulário de Upload AJAX -->
                    <div class="mb-4">
                        <div class="upload-area" id="uploadArea">
                            <div class="upload-icon">
                                <i class="fas fa-cloud-upload-alt fa-4x text-primary"></i>
                            </div>
                            <h4 class="upload-text mt-3">Arraste e solte arquivos aqui</h4>
                            <p class="text-muted">ou</p>
                            <button type="button" class="btn btn-primary" id="fileSelector">
                                <i class="fas fa-file-upload me-2"></i> Selecionar Arquivos
                            </button>
                            <input type="file" hidden id="fileInput">
                        </div>
                    </div>
                    
                    <!-- Configurações do upload -->
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="visibilidade" class="form-label">Visibilidade</label>
                                <select class="form-select" id="visibilidade">
                                    <option value="privado">Privado (apenas você)</option>
                                    <option value="publico">Público (todos da franquia)</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="franquia" class="form-label">Franquia (opcional)</label>
                                <select class="form-select" id="franquia">
                                    <option value="">Pessoal (nenhuma franquia)</option>
                                    <?php foreach ($franquias_usuario as $fr): ?>
                                    <option value="<?php echo $fr['id']; ?>" <?php echo ($pasta_atual && $pasta_atual['franquia_id'] == $fr['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($fr['nome']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Lista de arquivos a serem enviados -->
                    <div class="mt-4" id="fileList" style="display: none;">
                        <h6 class="mb-3"><i class="fas fa-list me-2"></i> Arquivos a serem enviados:</h6>
                        <div class="list-group" id="fileListItems">
                            <!-- O JavaScript irá preencher isso -->
                        </div>
                        <div class="mt-3">
                            <button type="button" class="btn btn-success" id="uploadBtn">
                                <i class="fas fa-upload me-2"></i> Iniciar Upload
                            </button>
                            <button type="button" class="btn btn-outline-danger" id="clearBtn">
                                <i class="fas fa-times me-2"></i> Limpar
                            </button>
                        </div>
                    </div>
                    
                    <!-- Progresso do upload -->
                    <div class="mt-4" id="uploadProgress" style="display: none;">
                        <h6 class="mb-3"><i class="fas fa-spinner me-2"></i> Progresso:</h6>
                        <div class="progress">
                            <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                        </div>
                        <p class="text-center mt-2" id="progressText">0%</p>
                    </div>
                    
                    <!-- Formulário tradicional (fallback) -->
                    <div class="mt-5">
                        <hr>
                        <h6 class="mb-3">Método tradicional:</h6>
                        <form action="" method="post" enctype="multipart/form-data">
                            <?php if ($pasta_id): ?>
                            <input type="hidden" name="pasta_id" value="<?php echo $pasta_id; ?>">
                            <?php endif; ?>
                            
                            <div class="mb-3">
                                <label for="arquivo" class="form-label">Selecione o arquivo</label>
                                <input class="form-control" type="file" id="arquivo" name="arquivo" required>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="visibilidade_tradicional" class="form-label">Visibilidade</label>
                                        <select class="form-select" id="visibilidade_tradicional" name="visibilidade">
                                            <option value="privado">Privado (apenas você)</option>
                                            <option value="publico">Público (todos da franquia)</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="franquia_tradicional" class="form-label">Franquia (opcional)</label>
                                        <select class="form-select" id="franquia_tradicional" name="franquia_id">
                                            <option value="">Pessoal (nenhuma franquia)</option>
                                            <?php foreach ($franquias_usuario as $fr): ?>
                                            <option value="<?php echo $fr['id']; ?>" <?php echo ($pasta_atual && $pasta_atual['franquia_id'] == $fr['id']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($fr['nome']); ?>
                                            </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="text-end">
                                <a href="/arquivos.php<?php echo $pasta_id ? '?pasta=' . $pasta_id : ''; ?>" class="btn btn-secondary">Cancelar</a>
                                <button type="submit" class="btn btn-primary">Enviar Arquivo</button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
    .upload-area {
        border: 2px dashed #0d6efd;
        border-radius: 10px;
        padding: 50px 20px;
        text-align: center;
        transition: all 0.3s;
        background-color: #f8f9fa;
    }
    .upload-area:hover, .upload-area.active {
        background-color: #e9ecef;
        cursor: pointer;
    }
    .upload-icon {
        margin-bottom: 15px;
    }
    .file-item {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 8px 15px;
    }
    .file-item .file-name {
        display: flex;
        align-items: center;
    }
    .file-item .file-icon {
        margin-right: 10px;
        font-size: 1.2em;
    }
    .file-item .file-remove {
        cursor: pointer;
        color: #dc3545;
    }
    .file-item .file-remove:hover {
        color: #b02a37;
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Elementos DOM
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('fileInput');
        const fileSelector = document.getElementById('fileSelector');
        const fileList = document.getElementById('fileList');
        const fileListItems = document.getElementById('fileListItems');
        const uploadBtn = document.getElementById('uploadBtn');
        const clearBtn = document.getElementById('clearBtn');
        const uploadProgress = document.getElementById('uploadProgress');
        const progressBar = uploadProgress.querySelector('.progress-bar');
        const progressText = document.getElementById('progressText');
        const visibilidadeSelect = document.getElementById('visibilidade');
        const franquiaSelect = document.getElementById('franquia');
        
        // Arquivos selecionados
        let selectedFiles = [];
        
        // Pasta atual (se houver)
        const pastaId = <?php echo $pasta_id ? $pasta_id : 'null'; ?>;
        
        // Eventos para drag and drop
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadArea.addEventListener(eventName, highlight, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, unhighlight, false);
        });
        
        function highlight() {
            uploadArea.classList.add('active');
        }
        
        function unhighlight() {
            uploadArea.classList.remove('active');
        }
        
        // Evento de soltar arquivos
        uploadArea.addEventListener('drop', handleDrop, false);
        
        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;
            handleFiles(files);
        }
        
        // Evento de seleção de arquivos pelo botão
        fileSelector.addEventListener('click', () => {
            fileInput.click();
        });
        
        fileInput.addEventListener('change', () => {
            handleFiles(fileInput.files);
        });
        
        // Processar arquivos selecionados
        function handleFiles(files) {
            if (files.length === 0) return;
            
            // Converter FileList para array e adicionar aos selecionados
            Array.from(files).forEach(file => {
                // Verificar se já existe na lista
                const exists = selectedFiles.some(f => 
                    f.name === file.name && 
                    f.size === file.size && 
                    f.lastModified === file.lastModified
                );
                
                if (!exists) {
                    selectedFiles.push(file);
                }
            });
            
            updateFileList();
            
            // Limpar input para permitir selecionar o mesmo arquivo novamente
            fileInput.value = '';
        }
        
        // Atualizar lista de arquivos
        function updateFileList() {
            // Mostrar ou esconder a lista
            if (selectedFiles.length > 0) {
                fileList.style.display = 'block';
            } else {
                fileList.style.display = 'none';
            }
            
            // Limpar lista atual
            fileListItems.innerHTML = '';
            
            // Adicionar cada arquivo
            selectedFiles.forEach((file, index) => {
                const item = document.createElement('div');
                item.className = 'list-group-item file-item';
                
                // Determinar ícone com base no tipo de arquivo
                let iconClass = 'fa-file';
                const fileExt = file.name.split('.').pop().toLowerCase();
                
                if (['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'].includes(fileExt)) {
                    iconClass = 'fa-file-image';
                } else if (['mp4', 'webm', 'avi', 'mov', 'wmv'].includes(fileExt)) {
                    iconClass = 'fa-file-video';
                } else if (fileExt === 'pdf') {
                    iconClass = 'fa-file-pdf';
                } else if (['doc', 'docx'].includes(fileExt)) {
                    iconClass = 'fa-file-word';
                } else if (['xls', 'xlsx'].includes(fileExt)) {
                    iconClass = 'fa-file-excel';
                } else if (['ppt', 'pptx'].includes(fileExt)) {
                    iconClass = 'fa-file-powerpoint';
                }
                
                item.innerHTML = `
                    <div class="file-name">
                        <span class="file-icon"><i class="fas ${iconClass}"></i></span>
                        ${file.name} <small class="text-muted ms-2">(${formatFileSize(file.size)})</small>
                    </div>
                    <span class="file-remove" data-index="${index}"><i class="fas fa-times"></i></span>
                `;
                
                fileListItems.appendChild(item);
            });
            
            // Adicionar eventos para os botões de remover
            document.querySelectorAll('.file-remove').forEach(btn => {
                btn.addEventListener('click', function() {
                    const index = parseInt(this.getAttribute('data-index'));
                    selectedFiles.splice(index, 1);
                    updateFileList();
                });
            });
        }
        
        // Função para formatar tamanho do arquivo
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // Botão para limpar a lista
        clearBtn.addEventListener('click', () => {
            selectedFiles = [];
            updateFileList();
        });
        
        // Botão para iniciar o upload
        uploadBtn.addEventListener('click', uploadFiles);
        
        // Função para fazer upload
        function uploadFiles() {
            if (selectedFiles.length === 0) return;
            
            // Mostrar barra de progresso
            uploadProgress.style.display = 'block';
            
            // Obter configurações
            const visibilidade = visibilidadeSelect.value;
            const franquiaId = franquiaSelect.value;
            
            // Contador para controlar o progresso
            let completed = 0;
            const total = selectedFiles.length;
            
            // Desabilitar botões durante o upload
            uploadBtn.disabled = true;
            clearBtn.disabled = true;
            
            // Fazer upload de cada arquivo
            selectedFiles.forEach(file => {
                const formData = new FormData();
                formData.append('arquivo', file);
                formData.append('visibilidade', visibilidade);
                
                if (franquiaId) {
                    formData.append('franquia_id', franquiaId);
                }
                
                if (pastaId) {
                    formData.append('pasta_id', pastaId);
                }
                
                fetch('/api/arquivos/upload', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    completed++;
                    updateProgress(completed, total);
                    
                    if (completed === total) {
                        finishUpload();
                    }
                })
                .catch(error => {
                    console.error('Erro ao fazer upload:', error);
                    completed++;
                    updateProgress(completed, total);
                    
                    if (completed === total) {
                        finishUpload();
                    }
                });
            });
        }
        
        // Atualizar a barra de progresso
        function updateProgress(completed, total) {
            const percentage = Math.round((completed / total) * 100);
            progressBar.style.width = percentage + '%';
            progressText.textContent = percentage + '%';
        }
        
        // Finalizar o processo de upload
        function finishUpload() {
            setTimeout(() => {
                alert('Upload concluído!');
                
                // Redirecionar para a página de arquivos
                if (pastaId) {
                    window.location.href = `/arquivos.php?pasta=${pastaId}`;
                } else {
                    window.location.href = '/arquivos.php';
                }
            }, 1000);
        }
    });
</script>

<?php
// Capturar o conteúdo e armazenar na variável
$conteudo = ob_get_clean();

// Incluir o layout principal
include_once APP_PATH . '/views/layout.php';
?> 