<?php
class Usuario {
    private $db;

    public function __construct() {
        $this->db = db_connect();
    }

    /**
     * Obter todos os usuários
     */
    public function getAll() {
        $stmt = $this->db->query('SELECT * FROM usuarios ORDER BY nome');
        return $stmt->fetchAll();
    }

    /**
     * Obter usuário por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare('SELECT * FROM usuarios WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obter usuário por email
     */
    public function getByEmail($email) {
        $stmt = $this->db->prepare('SELECT * FROM usuarios WHERE email = ?');
        $stmt->execute([$email]);
        return $stmt->fetch();
    }

    /**
     * Obter usuários por tipo
     */
    public function getByTipo($tipo) {
        $stmt = $this->db->prepare('SELECT * FROM usuarios WHERE tipo = ? ORDER BY nome');
        $stmt->execute([$tipo]);
        return $stmt->fetchAll();
    }

    /**
     * Obter usuários por franquia
     */
    public function getByFranquia($franquia_id) {
        $stmt = $this->db->prepare('
            SELECT u.* FROM usuarios u
            JOIN usuario_franquia uf ON u.id = uf.usuario_id
            WHERE uf.franquia_id = ?
            ORDER BY u.nome
        ');
        $stmt->execute([$franquia_id]);
        return $stmt->fetchAll();
    }

    /**
     * Criar um novo usuário
     */
    public function create($data) {
        $stmt = $this->db->prepare('
            INSERT INTO usuarios (nome, email, senha, tipo, status)
            VALUES (?, ?, ?, ?, ?)
        ');
        
        // Hash da senha
        $senha_hash = password_hash($data['senha'], PASSWORD_DEFAULT);
        
        return $stmt->execute([
            $data['nome'],
            $data['email'],
            $senha_hash,
            $data['tipo'],
            isset($data['status']) ? $data['status'] : 1
        ]);
    }

    /**
     * Atualizar um usuário existente
     */
    public function update($id, $data) {
        // Se a senha estiver vazia, não a atualizamos
        if (empty($data['senha'])) {
            $stmt = $this->db->prepare('
                UPDATE usuarios 
                SET nome = ?, email = ?, tipo = ?, status = ?
                WHERE id = ?
            ');
            return $stmt->execute([
                $data['nome'],
                $data['email'],
                $data['tipo'],
                isset($data['status']) ? $data['status'] : 1,
                $id
            ]);
        } else {
            $stmt = $this->db->prepare('
                UPDATE usuarios 
                SET nome = ?, email = ?, senha = ?, tipo = ?, status = ?
                WHERE id = ?
            ');
            
            // Hash da senha
            $senha_hash = password_hash($data['senha'], PASSWORD_DEFAULT);
            
            return $stmt->execute([
                $data['nome'],
                $data['email'],
                $senha_hash,
                $data['tipo'],
                isset($data['status']) ? $data['status'] : 1,
                $id
            ]);
        }
    }

    /**
     * Excluir um usuário
     */
    public function delete($id) {
        $stmt = $this->db->prepare('DELETE FROM usuarios WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Verificar credenciais de login
     */
    public function verificarLogin($email, $senha) {
        $usuario = $this->getByEmail($email);
        
        if ($usuario && password_verify($senha, $usuario['senha']) && $usuario['status'] == 1) {
            return $usuario;
        }
        
        return false;
    }

    /**
     * Adicionar usuário a uma franquia
     */
    public function adicionarAFranquia($usuario_id, $franquia_id) {
        $stmt = $this->db->prepare('
            INSERT INTO usuario_franquia (usuario_id, franquia_id)
            VALUES (?, ?)
        ');
        return $stmt->execute([$usuario_id, $franquia_id]);
    }

    /**
     * Remover usuário de uma franquia
     */
    public function removerDeFranquia($usuario_id, $franquia_id) {
        $stmt = $this->db->prepare('
            DELETE FROM usuario_franquia 
            WHERE usuario_id = ? AND franquia_id = ?
        ');
        return $stmt->execute([$usuario_id, $franquia_id]);
    }

    /**
     * Obter franquias do usuário
     */
    public function getFranquias($usuario_id) {
        $stmt = $this->db->prepare('
            SELECT f.* FROM franquias f
            JOIN usuario_franquia uf ON f.id = uf.franquia_id
            WHERE uf.usuario_id = ?
        ');
        $stmt->execute([$usuario_id]);
        return $stmt->fetchAll();
    }

    /**
     * Alterar senha do usuário
     */
    public function alterarSenha($id, $nova_senha) {
        $senha_hash = password_hash($nova_senha, PASSWORD_DEFAULT);
        
        $stmt = $this->db->prepare('
            UPDATE usuarios 
            SET senha = ?
            WHERE id = ?
        ');
        
        return $stmt->execute([$senha_hash, $id]);
    }
} 