<?php
class Pasta {
    private $db;

    public function __construct() {
        $this->db = db_connect();
    }

    /**
     * Obter todas as pastas
     */
    public function getAll($usuario_id = null, $franquia_id = null) {
        if ($usuario_id && $franquia_id) {
            $stmt = $this->db->prepare('
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE p.usuario_id = ? OR p.franquia_id = ?
                ORDER BY p.nome
            ');
            $stmt->execute([$usuario_id, $franquia_id]);
        } elseif ($usuario_id) {
            $stmt = $this->db->prepare('
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE p.usuario_id = ?
                ORDER BY p.nome
            ');
            $stmt->execute([$usuario_id]);
        } elseif ($franquia_id) {
            $stmt = $this->db->prepare('
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE p.franquia_id = ?
                ORDER BY p.nome
            ');
            $stmt->execute([$franquia_id]);
        } else {
            $stmt = $this->db->query('
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                ORDER BY p.nome
            ');
        }
        
        return $stmt->fetchAll();
    }

    /**
     * Obter pasta por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare('
            SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM pastas p
            LEFT JOIN usuarios u ON p.usuario_id = u.id
            LEFT JOIN franquias f ON p.franquia_id = f.id
            WHERE p.id = ?
        ');
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obter subpastas de uma pasta
     */
    public function getSubpastas($pasta_pai_id = null, $usuario_id = null, $franquia_id = null) {
        // Se pasta_pai_id for null, buscamos as pastas raiz
        $sql_pasta_pai = ($pasta_pai_id === null) ? 'p.pasta_pai_id IS NULL' : 'p.pasta_pai_id = ?';
        $params = [];
        
        if ($pasta_pai_id !== null) {
            $params[] = $pasta_pai_id;
        }
        
        if ($usuario_id && $franquia_id) {
            $sql = "
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE $sql_pasta_pai AND (p.usuario_id = ? OR p.franquia_id = ?)
                ORDER BY p.nome
            ";
            $params[] = $usuario_id;
            $params[] = $franquia_id;
        } elseif ($usuario_id) {
            $sql = "
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE $sql_pasta_pai AND p.usuario_id = ?
                ORDER BY p.nome
            ";
            $params[] = $usuario_id;
        } elseif ($franquia_id) {
            $sql = "
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE $sql_pasta_pai AND p.franquia_id = ?
                ORDER BY p.nome
            ";
            $params[] = $franquia_id;
        } else {
            $sql = "
                SELECT p.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM pastas p
                LEFT JOIN usuarios u ON p.usuario_id = u.id
                LEFT JOIN franquias f ON p.franquia_id = f.id
                WHERE $sql_pasta_pai
                ORDER BY p.nome
            ";
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    /**
     * Criar uma nova pasta
     */
    public function create($data) {
        $stmt = $this->db->prepare('
            INSERT INTO pastas (nome, pasta_pai_id, usuario_id, franquia_id)
            VALUES (?, ?, ?, ?)
        ');
        
        $resultado = $stmt->execute([
            $data['nome'],
            $data['pasta_pai_id'] ?? null,
            $data['usuario_id'],
            $data['franquia_id'] ?? null
        ]);
        
        if ($resultado) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }

    /**
     * Atualizar uma pasta existente
     */
    public function update($id, $data) {
        $stmt = $this->db->prepare('
            UPDATE pastas 
            SET nome = ?, pasta_pai_id = ?, franquia_id = ?
            WHERE id = ?
        ');
        
        return $stmt->execute([
            $data['nome'],
            $data['pasta_pai_id'] ?? null,
            $data['franquia_id'] ?? null,
            $id
        ]);
    }

    /**
     * Excluir uma pasta e seu conteúdo (arquivos e subpastas)
     */
    public function delete($id) {
        // Primeiro, verificamos se a pasta existe
        $pasta = $this->getById($id);
        if (!$pasta) {
            return false;
        }
        
        // Obtemos as subpastas
        $subpastas = $this->getSubpastas($id);
        
        // Excluímos recursivamente as subpastas
        foreach ($subpastas as $subpasta) {
            $this->delete($subpasta['id']);
        }
        
        // Obtemos os arquivos da pasta
        $arquivo = new Arquivo();
        $arquivos = $arquivo->getByPasta($id);
        
        // Removemos as relações com os arquivos (não excluímos os arquivos em si)
        $stmt = $this->db->prepare('DELETE FROM arquivo_pasta WHERE pasta_id = ?');
        $stmt->execute([$id]);
        
        // Por fim, removemos a pasta
        $stmt = $this->db->prepare('DELETE FROM pastas WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Verificar se uma pasta existe
     */
    public function exists($id) {
        $stmt = $this->db->prepare('SELECT COUNT(*) FROM pastas WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetchColumn() > 0;
    }

    /**
     * Verificar se uma pasta é ancestral de outra
     * (para evitar ciclos ao mover pastas)
     */
    public function isAncestor($pasta_id, $possivel_ancestral_id) {
        if ($pasta_id == $possivel_ancestral_id) {
            return true;
        }
        
        $pasta = $this->getById($pasta_id);
        if (!$pasta || !$pasta['pasta_pai_id']) {
            return false;
        }
        
        if ($pasta['pasta_pai_id'] == $possivel_ancestral_id) {
            return true;
        }
        
        return $this->isAncestor($pasta['pasta_pai_id'], $possivel_ancestral_id);
    }

    /**
     * Obter o caminho completo da pasta
     */
    public function getCaminho($id) {
        $caminho = [];
        $pasta_atual = $this->getById($id);
        
        while ($pasta_atual) {
            array_unshift($caminho, $pasta_atual);
            
            if ($pasta_atual['pasta_pai_id']) {
                $pasta_atual = $this->getById($pasta_atual['pasta_pai_id']);
            } else {
                $pasta_atual = null;
            }
        }
        
        return $caminho;
    }

    /**
     * Contar arquivos em uma pasta (incluindo subpastas)
     */
    public function contarArquivos($id) {
        // Contagem direta de arquivos na pasta
        $stmt = $this->db->prepare('
            SELECT COUNT(*) FROM arquivo_pasta WHERE pasta_id = ?
        ');
        $stmt->execute([$id]);
        $contagem = $stmt->fetchColumn();
        
        // Obtemos as subpastas
        $subpastas = $this->getSubpastas($id);
        
        // Somamos os arquivos das subpastas
        foreach ($subpastas as $subpasta) {
            $contagem += $this->contarArquivos($subpasta['id']);
        }
        
        return $contagem;
    }
} 