<?php
class Notificacao {
    private $db;

    public function __construct() {
        $this->db = db_connect();
    }

    /**
     * Obter todas as notificações
     */
    public function getAll() {
        $stmt = $this->db->query('
            SELECT n.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM notificacoes n
            LEFT JOIN usuarios u ON n.usuario_id = u.id
            LEFT JOIN franquias f ON n.franquia_id = f.id
            ORDER BY n.created_at DESC
        ');
        return $stmt->fetchAll();
    }

    /**
     * Obter notificação por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare('
            SELECT n.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM notificacoes n
            LEFT JOIN usuarios u ON n.usuario_id = u.id
            LEFT JOIN franquias f ON n.franquia_id = f.id
            WHERE n.id = ?
        ');
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obter notificações de um usuário
     */
    public function getByUsuario($usuario_id, $limit = null, $apenas_nao_lidas = false) {
        $sql = '
            SELECT n.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM notificacoes n
            LEFT JOIN usuarios u ON n.usuario_id = u.id
            LEFT JOIN franquias f ON n.franquia_id = f.id
            WHERE n.para_todos = 1 OR n.usuario_id = ?
        ';
        
        // Verificar também notificações da franquia do usuário
        $usuario = new Usuario();
        $franquias = $usuario->getFranquias($usuario_id);
        
        if (!empty($franquias)) {
            $franquia_ids = array_column($franquias, 'id');
            $placeholders = implode(',', array_fill(0, count($franquia_ids), '?'));
            $sql .= " OR n.franquia_id IN ($placeholders)";
        }
        
        if ($apenas_nao_lidas) {
            $sql .= ' AND n.lida = 0';
        }
        
        $sql .= ' ORDER BY n.created_at DESC';
        
        if ($limit) {
            $sql .= ' LIMIT ?';
        }
        
        $stmt = $this->db->prepare($sql);
        
        $params = [$usuario_id];
        
        if (!empty($franquias)) {
            foreach ($franquia_ids as $franquia_id) {
                $params[] = $franquia_id;
            }
        }
        
        if ($limit) {
            $params[] = $limit;
        }
        
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }

    /**
     * Criar uma nova notificação
     */
    public function create($data) {
        $stmt = $this->db->prepare('
            INSERT INTO notificacoes (titulo, mensagem, tipo, para_todos, usuario_id, franquia_id)
            VALUES (?, ?, ?, ?, ?, ?)
        ');
        
        return $stmt->execute([
            $data['titulo'],
            $data['mensagem'],
            $data['tipo'] ?? 'info',
            $data['para_todos'] ?? 0,
            $data['usuario_id'] ?? null,
            $data['franquia_id'] ?? null
        ]);
    }

    /**
     * Marcar notificação como lida
     */
    public function marcarComoLida($id) {
        $stmt = $this->db->prepare('UPDATE notificacoes SET lida = 1 WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Marcar todas as notificações de um usuário como lidas
     */
    public function marcarTodasComoLidas($usuario_id) {
        $stmt = $this->db->prepare('
            UPDATE notificacoes 
            SET lida = 1 
            WHERE usuario_id = ? OR para_todos = 1
        ');
        
        // Também marcar como lidas as notificações das franquias do usuário
        $usuario = new Usuario();
        $franquias = $usuario->getFranquias($usuario_id);
        
        $resultado = $stmt->execute([$usuario_id]);
        
        if ($resultado && !empty($franquias)) {
            $franquia_ids = array_column($franquias, 'id');
            $placeholders = implode(',', array_fill(0, count($franquia_ids), '?'));
            
            $stmt = $this->db->prepare("
                UPDATE notificacoes 
                SET lida = 1 
                WHERE franquia_id IN ($placeholders)
            ");
            
            $stmt->execute($franquia_ids);
        }
        
        return $resultado;
    }

    /**
     * Excluir uma notificação
     */
    public function delete($id) {
        $stmt = $this->db->prepare('DELETE FROM notificacoes WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Contar notificações não lidas de um usuário
     */
    public function contarNaoLidas($usuario_id) {
        $sql = '
            SELECT COUNT(*) 
            FROM notificacoes 
            WHERE lida = 0 AND (para_todos = 1 OR usuario_id = ?)
        ';
        
        // Verificar também notificações da franquia do usuário
        $usuario = new Usuario();
        $franquias = $usuario->getFranquias($usuario_id);
        
        $params = [$usuario_id];
        
        if (!empty($franquias)) {
            $franquia_ids = array_column($franquias, 'id');
            $placeholders = implode(',', array_fill(0, count($franquia_ids), '?'));
            $sql .= " OR franquia_id IN ($placeholders)";
            
            foreach ($franquia_ids as $franquia_id) {
                $params[] = $franquia_id;
            }
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchColumn();
    }

    /**
     * Enviar notificação para todos os usuários
     */
    public function enviarParaTodos($titulo, $mensagem, $tipo = 'info') {
        return $this->create([
            'titulo' => $titulo,
            'mensagem' => $mensagem,
            'tipo' => $tipo,
            'para_todos' => 1
        ]);
    }

    /**
     * Enviar notificação para um usuário específico
     */
    public function enviarParaUsuario($usuario_id, $titulo, $mensagem, $tipo = 'info') {
        return $this->create([
            'titulo' => $titulo,
            'mensagem' => $mensagem,
            'tipo' => $tipo,
            'para_todos' => 0,
            'usuario_id' => $usuario_id
        ]);
    }

    /**
     * Enviar notificação para uma franquia
     */
    public function enviarParaFranquia($franquia_id, $titulo, $mensagem, $tipo = 'info') {
        return $this->create([
            'titulo' => $titulo,
            'mensagem' => $mensagem,
            'tipo' => $tipo,
            'para_todos' => 0,
            'franquia_id' => $franquia_id
        ]);
    }
} 