<?php
class Franquia {
    private $db;

    public function __construct() {
        $this->db = db_connect();
    }

    /**
     * Obter todas as franquias
     */
    public function getAll() {
        $stmt = $this->db->query('SELECT * FROM franquias ORDER BY nome');
        return $stmt->fetchAll();
    }

    /**
     * Obter franquia por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare('SELECT * FROM franquias WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obter franquias disponíveis para um usuário
     * (Franquias às quais o usuário não está vinculado)
     */
    public function getDisponiveisParaUsuario($usuario_id) {
        $stmt = $this->db->prepare('
            SELECT f.* FROM franquias f
            WHERE f.id NOT IN (
                SELECT franquia_id FROM usuario_franquia WHERE usuario_id = ?
            )
            ORDER BY f.nome
        ');
        $stmt->execute([$usuario_id]);
        return $stmt->fetchAll();
    }

    /**
     * Criar uma nova franquia
     */
    public function create($data) {
        $stmt = $this->db->prepare('
            INSERT INTO franquias (nome, cnpj, endereco, telefone, email, status)
            VALUES (?, ?, ?, ?, ?, ?)
        ');
        
        return $stmt->execute([
            $data['nome'],
            $data['cnpj'],
            $data['endereco'],
            $data['telefone'],
            $data['email'],
            isset($data['status']) ? $data['status'] : 1
        ]);
    }

    /**
     * Atualizar uma franquia existente
     */
    public function update($id, $data) {
        $stmt = $this->db->prepare('
            UPDATE franquias 
            SET nome = ?, cnpj = ?, endereco = ?, telefone = ?, email = ?, status = ?
            WHERE id = ?
        ');
        
        return $stmt->execute([
            $data['nome'],
            $data['cnpj'],
            $data['endereco'],
            $data['telefone'],
            $data['email'],
            isset($data['status']) ? $data['status'] : 1,
            $id
        ]);
    }

    /**
     * Excluir uma franquia
     */
    public function delete($id) {
        $stmt = $this->db->prepare('DELETE FROM franquias WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Obter usuários de uma franquia
     */
    public function getUsuarios($franquia_id) {
        $stmt = $this->db->prepare('
            SELECT u.* FROM usuarios u
            JOIN usuario_franquia uf ON u.id = uf.usuario_id
            WHERE uf.franquia_id = ?
            ORDER BY u.nome
        ');
        $stmt->execute([$franquia_id]);
        return $stmt->fetchAll();
    }

    /**
     * Adicionar um usuário à franquia
     */
    public function adicionarUsuario($franquia_id, $usuario_id) {
        $stmt = $this->db->prepare('
            INSERT INTO usuario_franquia (franquia_id, usuario_id)
            VALUES (?, ?)
        ');
        return $stmt->execute([$franquia_id, $usuario_id]);
    }

    /**
     * Remover um usuário da franquia
     */
    public function removerUsuario($franquia_id, $usuario_id) {
        $stmt = $this->db->prepare('
            DELETE FROM usuario_franquia 
            WHERE franquia_id = ? AND usuario_id = ?
        ');
        return $stmt->execute([$franquia_id, $usuario_id]);
    }

    /**
     * Verificar se uma franquia existe
     */
    public function exists($id) {
        $stmt = $this->db->prepare('SELECT COUNT(*) FROM franquias WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetchColumn() > 0;
    }

    /**
     * Verificar se um CNPJ já está cadastrado
     */
    public function cnpjExists($cnpj, $excluir_id = null) {
        if ($excluir_id) {
            $stmt = $this->db->prepare('SELECT COUNT(*) FROM franquias WHERE cnpj = ? AND id != ?');
            $stmt->execute([$cnpj, $excluir_id]);
        } else {
            $stmt = $this->db->prepare('SELECT COUNT(*) FROM franquias WHERE cnpj = ?');
            $stmt->execute([$cnpj]);
        }
        return $stmt->fetchColumn() > 0;
    }
} 