<?php
class Evento {
    private $db;

    public function __construct() {
        $this->db = db_connect();
    }

    /**
     * Obter todos os eventos
     */
    public function getAll() {
        $stmt = $this->db->query('
            SELECT e.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM eventos e
            LEFT JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN franquias f ON e.franquia_id = f.id
            ORDER BY e.data_inicio ASC
        ');
        return $stmt->fetchAll();
    }

    /**
     * Obter evento por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare('
            SELECT e.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM eventos e
            LEFT JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN franquias f ON e.franquia_id = f.id
            WHERE e.id = ?
        ');
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obter eventos de um usuário
     */
    public function getByUsuario($usuario_id) {
        $sql = '
            SELECT e.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM eventos e
            LEFT JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN franquias f ON e.franquia_id = f.id
            WHERE e.para_todos = 1 OR e.usuario_id = ?
        ';
        
        // Verificar também eventos da franquia do usuário
        $usuario = new Usuario();
        $franquias = $usuario->getFranquias($usuario_id);
        
        $params = [$usuario_id];
        
        if (!empty($franquias)) {
            $franquia_ids = array_column($franquias, 'id');
            $placeholders = implode(',', array_fill(0, count($franquia_ids), '?'));
            $sql .= " OR e.franquia_id IN ($placeholders)";
            
            foreach ($franquia_ids as $franquia_id) {
                $params[] = $franquia_id;
            }
        }
        
        $sql .= ' ORDER BY e.data_inicio ASC';
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }

    /**
     * Obter eventos de uma franquia
     */
    public function getByFranquia($franquia_id) {
        $stmt = $this->db->prepare('
            SELECT e.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM eventos e
            LEFT JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN franquias f ON e.franquia_id = f.id
            WHERE e.para_todos = 1 OR e.franquia_id = ?
            ORDER BY e.data_inicio ASC
        ');
        $stmt->execute([$franquia_id]);
        return $stmt->fetchAll();
    }

    /**
     * Obter eventos em um período específico
     */
    public function getByPeriodo($data_inicio, $data_fim, $usuario_id = null, $franquia_id = null) {
        $sql = '
            SELECT e.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM eventos e
            LEFT JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN franquias f ON e.franquia_id = f.id
            WHERE (
                (e.data_inicio BETWEEN ? AND ?) OR
                (e.data_fim BETWEEN ? AND ?) OR
                (e.data_inicio <= ? AND e.data_fim >= ?)
            )
        ';
        
        $params = [$data_inicio, $data_fim, $data_inicio, $data_fim, $data_inicio, $data_fim];
        
        // Filtrar por usuário e/ou franquia
        if ($usuario_id && $franquia_id) {
            $sql .= ' AND (e.para_todos = 1 OR e.usuario_id = ? OR e.franquia_id = ?)';
            $params[] = $usuario_id;
            $params[] = $franquia_id;
        } elseif ($usuario_id) {
            $sql .= ' AND (e.para_todos = 1 OR e.usuario_id = ?)';
            $params[] = $usuario_id;
            
            // Verificar também eventos da franquia do usuário
            $usuario = new Usuario();
            $franquias = $usuario->getFranquias($usuario_id);
            
            if (!empty($franquias)) {
                $franquia_ids = array_column($franquias, 'id');
                $placeholders = implode(',', array_fill(0, count($franquia_ids), '?'));
                $sql .= " OR e.franquia_id IN ($placeholders)";
                
                foreach ($franquia_ids as $franquia_id) {
                    $params[] = $franquia_id;
                }
            }
        } elseif ($franquia_id) {
            $sql .= ' AND (e.para_todos = 1 OR e.franquia_id = ?)';
            $params[] = $franquia_id;
        }
        
        $sql .= ' ORDER BY e.data_inicio ASC';
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }

    /**
     * Criar um novo evento
     */
    public function create($data) {
        $stmt = $this->db->prepare('
            INSERT INTO eventos (titulo, descricao, data_inicio, data_fim, local, para_todos, usuario_id, franquia_id)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ');
        
        return $stmt->execute([
            $data['titulo'],
            $data['descricao'] ?? '',
            $data['data_inicio'],
            $data['data_fim'],
            $data['local'] ?? '',
            $data['para_todos'] ?? 0,
            $data['usuario_id'],
            $data['franquia_id'] ?? null
        ]);
    }

    /**
     * Atualizar um evento existente
     */
    public function update($id, $data) {
        $stmt = $this->db->prepare('
            UPDATE eventos 
            SET titulo = ?, descricao = ?, data_inicio = ?, data_fim = ?, 
                local = ?, para_todos = ?, franquia_id = ?
            WHERE id = ?
        ');
        
        return $stmt->execute([
            $data['titulo'],
            $data['descricao'] ?? '',
            $data['data_inicio'],
            $data['data_fim'],
            $data['local'] ?? '',
            $data['para_todos'] ?? 0,
            $data['franquia_id'] ?? null,
            $id
        ]);
    }

    /**
     * Excluir um evento
     */
    public function delete($id) {
        $stmt = $this->db->prepare('DELETE FROM eventos WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Obter próximos eventos
     */
    public function getProximos($limit = 5, $usuario_id = null, $franquia_id = null) {
        $data_atual = date('Y-m-d H:i:s');
        
        $sql = '
            SELECT e.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM eventos e
            LEFT JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN franquias f ON e.franquia_id = f.id
            WHERE e.data_inicio >= ?
        ';
        
        $params = [$data_atual];
        
        // Filtrar por usuário e/ou franquia
        if ($usuario_id && $franquia_id) {
            $sql .= ' AND (e.para_todos = 1 OR e.usuario_id = ? OR e.franquia_id = ?)';
            $params[] = $usuario_id;
            $params[] = $franquia_id;
        } elseif ($usuario_id) {
            $sql .= ' AND (e.para_todos = 1 OR e.usuario_id = ?)';
            $params[] = $usuario_id;
            
            // Verificar também eventos da franquia do usuário
            $usuario = new Usuario();
            $franquias = $usuario->getFranquias($usuario_id);
            
            if (!empty($franquias)) {
                $franquia_ids = array_column($franquias, 'id');
                $placeholders = implode(',', array_fill(0, count($franquia_ids), '?'));
                $sql .= " OR e.franquia_id IN ($placeholders)";
                
                foreach ($franquia_ids as $franquia_id) {
                    $params[] = $franquia_id;
                }
            }
        } elseif ($franquia_id) {
            $sql .= ' AND (e.para_todos = 1 OR e.franquia_id = ?)';
            $params[] = $franquia_id;
        }
        
        $sql .= ' ORDER BY e.data_inicio ASC LIMIT ?';
        $params[] = $limit;
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }

    /**
     * Verificar se um evento existe
     */
    public function exists($id) {
        $stmt = $this->db->prepare('SELECT COUNT(*) FROM eventos WHERE id = ?');
        $stmt->execute([$id]);
        return $stmt->fetchColumn() > 0;
    }

    /**
     * Formatar evento para o calendário (formato JSON)
     */
    public function formatarParaCalendario($eventos) {
        $resultado = [];
        
        foreach ($eventos as $evento) {
            $resultado[] = [
                'id' => $evento['id'],
                'title' => $evento['titulo'],
                'start' => $evento['data_inicio'],
                'end' => $evento['data_fim'],
                'description' => $evento['descricao'],
                'location' => $evento['local'],
                'allDay' => false,
                'className' => $this->getEventoClass($evento)
            ];
        }
        
        return $resultado;
    }
    
    /**
     * Obter classe CSS para o evento (baseado no tipo)
     */
    private function getEventoClass($evento) {
        if ($evento['para_todos']) {
            return 'evento-global';
        } elseif ($evento['franquia_id']) {
            return 'evento-franquia';
        } else {
            return 'evento-usuario';
        }
    }
} 