<?php
class Arquivo {
    private $db;

    public function __construct() {
        $this->db = db_connect();
    }

    /**
     * Obter todos os arquivos
     */
    public function getAll($usuario_id = null, $franquia_id = null) {
        if ($usuario_id && $franquia_id) {
            // Arquivos do usuário ou da franquia
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.usuario_id = ? OR a.franquia_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$usuario_id, $franquia_id]);
        } elseif ($usuario_id) {
            // Apenas arquivos do usuário
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.usuario_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$usuario_id]);
        } elseif ($franquia_id) {
            // Apenas arquivos da franquia
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.franquia_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$franquia_id]);
        } else {
            // Todos os arquivos
            $stmt = $this->db->query('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                ORDER BY a.created_at DESC
            ');
        }
        
        return $stmt->fetchAll();
    }

    /**
     * Obter arquivo por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare('
            SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM arquivos a
            LEFT JOIN usuarios u ON a.usuario_id = u.id
            LEFT JOIN franquias f ON a.franquia_id = f.id
            WHERE a.id = ?
        ');
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obter arquivos por pasta
     */
    public function getByPasta($pasta_id) {
        $stmt = $this->db->prepare('
            SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
            FROM arquivos a
            LEFT JOIN usuarios u ON a.usuario_id = u.id
            LEFT JOIN franquias f ON a.franquia_id = f.id
            JOIN arquivo_pasta ap ON a.id = ap.arquivo_id
            WHERE ap.pasta_id = ?
            ORDER BY a.nome
        ');
        $stmt->execute([$pasta_id]);
        return $stmt->fetchAll();
    }

    /**
     * Obter arquivos por tipo
     */
    public function getByTipo($tipo, $usuario_id = null, $franquia_id = null) {
        if ($usuario_id && $franquia_id) {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.tipo = ? AND (a.usuario_id = ? OR a.franquia_id = ?)
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$tipo, $usuario_id, $franquia_id]);
        } elseif ($usuario_id) {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.tipo = ? AND a.usuario_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$tipo, $usuario_id]);
        } elseif ($franquia_id) {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.tipo = ? AND a.franquia_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$tipo, $franquia_id]);
        } else {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.tipo = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute([$tipo]);
        }
        
        return $stmt->fetchAll();
    }

    /**
     * Criar um novo arquivo
     */
    public function create($data) {
        $stmt = $this->db->prepare('
            INSERT INTO arquivos (nome, tipo, categoria, caminho, tamanho, visibilidade, usuario_id, franquia_id)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ');
        
        $resultado = $stmt->execute([
            $data['nome'],
            $data['tipo'],
            $data['categoria'],
            $data['caminho'],
            $data['tamanho'],
            $data['visibilidade'] ?? 'privado',
            $data['usuario_id'],
            $data['franquia_id'] ?? null
        ]);
        
        if ($resultado) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }

    /**
     * Atualizar um arquivo existente
     */
    public function update($id, $data) {
        $stmt = $this->db->prepare('
            UPDATE arquivos 
            SET nome = ?, visibilidade = ?, franquia_id = ?
            WHERE id = ?
        ');
        
        return $stmt->execute([
            $data['nome'],
            $data['visibilidade'] ?? 'privado',
            $data['franquia_id'] ?? null,
            $id
        ]);
    }

    /**
     * Excluir um arquivo
     */
    public function delete($id) {
        // Primeiro, verificamos se o arquivo existe
        $arquivo = $this->getById($id);
        if (!$arquivo) {
            return false;
        }
        
        // Removemos o arquivo físico
        $caminho_fisico = UPLOAD_PATH . '/' . $arquivo['caminho'];
        if (file_exists($caminho_fisico)) {
            unlink($caminho_fisico);
        }
        
        // Removemos as relações com pastas
        $stmt = $this->db->prepare('DELETE FROM arquivo_pasta WHERE arquivo_id = ?');
        $stmt->execute([$id]);
        
        // Por fim, removemos o registro do banco
        $stmt = $this->db->prepare('DELETE FROM arquivos WHERE id = ?');
        return $stmt->execute([$id]);
    }

    /**
     * Adicionar arquivo a uma pasta
     */
    public function adicionarAPasta($arquivo_id, $pasta_id) {
        $stmt = $this->db->prepare('
            INSERT INTO arquivo_pasta (arquivo_id, pasta_id)
            VALUES (?, ?)
        ');
        return $stmt->execute([$arquivo_id, $pasta_id]);
    }

    /**
     * Remover arquivo de uma pasta
     */
    public function removerDaPasta($arquivo_id, $pasta_id) {
        $stmt = $this->db->prepare('
            DELETE FROM arquivo_pasta 
            WHERE arquivo_id = ? AND pasta_id = ?
        ');
        return $stmt->execute([$arquivo_id, $pasta_id]);
    }

    /**
     * Processar upload de arquivo
     */
    public function processarUpload($arquivo, $data) {
        if ($arquivo['error'] !== UPLOAD_ERR_OK) {
            return [
                'sucesso' => false,
                'mensagem' => 'Erro no upload do arquivo: ' . $this->getUploadErrorMessage($arquivo['error'])
            ];
        }
        
        // Verificar tamanho do arquivo
        if ($arquivo['size'] > MAX_UPLOAD_SIZE) {
            return [
                'sucesso' => false,
                'mensagem' => 'O arquivo excede o tamanho máximo permitido de ' . format_file_size(MAX_UPLOAD_SIZE)
            ];
        }
        
        // Determinar o tipo de arquivo e categoria
        $info = pathinfo($arquivo['name']);
        $extensao = strtolower($info['extension']);
        $categoria = $this->determinarCategoria($extensao);
        
        if (!$categoria) {
            return [
                'sucesso' => false,
                'mensagem' => 'Tipo de arquivo não permitido. Extensões permitidas: ' . 
                    implode(', ', array_merge(
                        ALLOWED_EXTENSIONS['documento'], 
                        ALLOWED_EXTENSIONS['imagem'], 
                        ALLOWED_EXTENSIONS['video']
                    ))
            ];
        }
        
        // Gerar nome único para o arquivo
        $nome_arquivo = uniqid() . '.' . $extensao;
        $caminho_relativo = $categoria . 's/' . $nome_arquivo;
        $caminho_destino = UPLOAD_PATH . '/' . $caminho_relativo;
        
        // Mover o arquivo para o diretório de destino
        if (!move_uploaded_file($arquivo['tmp_name'], $caminho_destino)) {
            return [
                'sucesso' => false,
                'mensagem' => 'Erro ao mover o arquivo para o diretório de destino.'
            ];
        }
        
        // Registrar o arquivo no banco de dados
        $arquivo_data = [
            'nome' => $info['filename'],
            'tipo' => $extensao,
            'categoria' => $categoria,
            'caminho' => $caminho_relativo,
            'tamanho' => $arquivo['size'],
            'visibilidade' => $data['visibilidade'] ?? 'privado',
            'usuario_id' => $data['usuario_id'],
            'franquia_id' => $data['franquia_id'] ?? null
        ];
        
        $arquivo_id = $this->create($arquivo_data);
        
        if (!$arquivo_id) {
            // Se falhar ao registrar no banco, remover o arquivo físico
            unlink($caminho_destino);
            return [
                'sucesso' => false,
                'mensagem' => 'Erro ao registrar o arquivo no banco de dados.'
            ];
        }
        
        // Se uma pasta foi especificada, adicionar o arquivo a ela
        if (isset($data['pasta_id']) && $data['pasta_id']) {
            $this->adicionarAPasta($arquivo_id, $data['pasta_id']);
        }
        
        return [
            'sucesso' => true,
            'mensagem' => 'Arquivo enviado com sucesso.',
            'arquivo_id' => $arquivo_id,
            'arquivo' => $this->getById($arquivo_id)
        ];
    }

    /**
     * Determinar a categoria do arquivo com base na extensão
     */
    private function determinarCategoria($extensao) {
        foreach (ALLOWED_EXTENSIONS as $categoria => $extensoes) {
            if (in_array($extensao, $extensoes)) {
                return $categoria;
            }
        }
        return false;
    }

    /**
     * Obter mensagem de erro de upload
     */
    private function getUploadErrorMessage($error_code) {
        switch ($error_code) {
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                return 'O arquivo excede o tamanho máximo permitido.';
            case UPLOAD_ERR_PARTIAL:
                return 'O upload do arquivo foi feito parcialmente.';
            case UPLOAD_ERR_NO_FILE:
                return 'Nenhum arquivo foi enviado.';
            case UPLOAD_ERR_NO_TMP_DIR:
                return 'Está faltando uma pasta temporária.';
            case UPLOAD_ERR_CANT_WRITE:
                return 'Falha ao gravar arquivo em disco.';
            case UPLOAD_ERR_EXTENSION:
                return 'Uma extensão PHP interrompeu o upload do arquivo.';
            default:
                return 'Erro desconhecido.';
        }
    }

    /**
     * Buscar arquivos
     */
    public function buscar($termo, $usuario_id = null, $franquia_id = null) {
        if ($usuario_id && $franquia_id) {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE (a.nome LIKE ? OR a.tipo LIKE ?) AND (a.usuario_id = ? OR a.franquia_id = ?)
                ORDER BY a.created_at DESC
            ');
            $stmt->execute(["%$termo%", "%$termo%", $usuario_id, $franquia_id]);
        } elseif ($usuario_id) {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE (a.nome LIKE ? OR a.tipo LIKE ?) AND a.usuario_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute(["%$termo%", "%$termo%", $usuario_id]);
        } elseif ($franquia_id) {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE (a.nome LIKE ? OR a.tipo LIKE ?) AND a.franquia_id = ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute(["%$termo%", "%$termo%", $franquia_id]);
        } else {
            $stmt = $this->db->prepare('
                SELECT a.*, u.nome as usuario_nome, f.nome as franquia_nome
                FROM arquivos a
                LEFT JOIN usuarios u ON a.usuario_id = u.id
                LEFT JOIN franquias f ON a.franquia_id = f.id
                WHERE a.nome LIKE ? OR a.tipo LIKE ?
                ORDER BY a.created_at DESC
            ');
            $stmt->execute(["%$termo%", "%$termo%"]);
        }
        
        return $stmt->fetchAll();
    }

    /**
     * Obter arquivos por usuário com pasta opcional
     */
    public function getByUsuario($usuario_id, $pasta = null) {
        $sql = '
            SELECT a.*, p.nome as pasta_nome
            FROM arquivos a
            LEFT JOIN arquivo_pasta ap ON a.id = ap.arquivo_id
            LEFT JOIN pastas p ON ap.pasta_id = p.id
            WHERE a.usuario_id = ?
        ';
        
        $params = [$usuario_id];
        
        if ($pasta !== null) {
            // Se pasta for vazia, busca arquivos sem pasta
            if (empty($pasta)) {
                $sql .= ' AND ap.pasta_id IS NULL';
            } else {
                // Buscar pasta pelo caminho
                $pasta_model = new Pasta();
                $pastas = $pasta_model->getAll($usuario_id);
                
                $pasta_ids = [];
                foreach ($pastas as $p) {
                    // Verifica se o caminho da pasta corresponde
                    $caminho = $this->getCaminhoPasta($p['id'], $pastas);
                    if ($caminho === $pasta) {
                        $pasta_ids[] = $p['id'];
                    }
                }
                
                if (!empty($pasta_ids)) {
                    $placeholders = implode(',', array_fill(0, count($pasta_ids), '?'));
                    $sql .= " AND ap.pasta_id IN ($placeholders)";
                    $params = array_merge($params, $pasta_ids);
                } else {
                    // Se não encontrou a pasta, retorna vazio
                    return [];
                }
            }
        }
        
        $sql .= ' ORDER BY a.created_at DESC';
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }

    /**
     * Obter arquivos por franquia com pasta opcional
     */
    public function getByFranquia($franquia_id, $pasta = null) {
        $sql = '
            SELECT a.*, p.nome as pasta_nome
            FROM arquivos a
            LEFT JOIN arquivo_pasta ap ON a.id = ap.arquivo_id
            LEFT JOIN pastas p ON ap.pasta_id = p.id
            WHERE a.franquia_id = ? AND (a.visibilidade = "publico" OR p.franquia_id = ?)
        ';
        
        $params = [$franquia_id, $franquia_id];
        
        if ($pasta !== null) {
            // Se pasta for vazia, busca arquivos sem pasta
            if (empty($pasta)) {
                $sql .= ' AND ap.pasta_id IS NULL';
            } else {
                // Buscar pasta pelo caminho
                $pasta_model = new Pasta();
                $pastas = $pasta_model->getAll(null, $franquia_id);
                
                $pasta_ids = [];
                foreach ($pastas as $p) {
                    // Verifica se o caminho da pasta corresponde
                    $caminho = $this->getCaminhoPasta($p['id'], $pastas);
                    if ($caminho === $pasta) {
                        $pasta_ids[] = $p['id'];
                    }
                }
                
                if (!empty($pasta_ids)) {
                    $placeholders = implode(',', array_fill(0, count($pasta_ids), '?'));
                    $sql .= " AND ap.pasta_id IN ($placeholders)";
                    $params = array_merge($params, $pasta_ids);
                } else {
                    // Se não encontrou a pasta, retorna vazio
                    return [];
                }
            }
        }
        
        $sql .= ' ORDER BY a.created_at DESC';
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll();
    }

    /**
     * Função auxiliar para obter o caminho completo da pasta
     */
    private function getCaminhoPasta($pasta_id, $todas_pastas) {
        $pasta_atual = null;
        foreach ($todas_pastas as $p) {
            if ($p['id'] == $pasta_id) {
                $pasta_atual = $p;
                break;
            }
        }
        
        if (!$pasta_atual) {
            return '';
        }
        
        $caminho = [$pasta_atual['nome']];
        $pasta_pai_id = $pasta_atual['pasta_pai_id'];
        
        while ($pasta_pai_id) {
            $encontrou = false;
            foreach ($todas_pastas as $p) {
                if ($p['id'] == $pasta_pai_id) {
                    array_unshift($caminho, $p['nome']);
                    $pasta_pai_id = $p['pasta_pai_id'];
                    $encontrou = true;
                    break;
                }
            }
            
            if (!$encontrou) {
                break;
            }
        }
        
        return implode('/', $caminho);
    }
} 