<?php
/**
 * Configurações globais do sistema
 */
session_start();

// Define constantes principais
define('BASE_PATH', dirname(dirname(__DIR__)));
define('APP_PATH', BASE_PATH . '/app');
define('PUBLIC_PATH', BASE_PATH . '/public');
define('UPLOAD_PATH', BASE_PATH . '/uploads');
define('URL', 'http://localhost/sistema-franqueados');

// Configurações de upload
define('MAX_UPLOAD_SIZE', 50 * 1024 * 1024); // 50MB
define('ALLOWED_EXTENSIONS', [
    'documento' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt'],
    'imagem' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
    'video' => ['mp4', 'webm', 'avi', 'mov', 'wmv']
]);

// Autoload das classes
spl_autoload_register(function ($class) {
    $prefix = '';
    $base_dir = APP_PATH;
    
    $file = $base_dir . '/' . str_replace('\\', '/', $class) . '.php';
    
    if (file_exists($file)) {
        require $file;
        return true;
    }
    
    return false;
});

// Carregar configurações do banco de dados
$db_config = require_once(APP_PATH . '/config/database.php');

// Função para conectar ao banco de dados
function db_connect() {
    global $db_config;
    try {
        $dsn = "mysql:host={$db_config['host']};dbname={$db_config['database']};charset={$db_config['charset']}";
        $pdo = new PDO($dsn, $db_config['username'], $db_config['password'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        die('Erro de conexão: ' . $e->getMessage());
    }
}

// Função para redirecionamento
function redirect($path) {
    header("Location: " . URL . $path);
    exit;
}

// Funções de controle de sessão/autenticação
function is_logged_in() {
    return isset($_SESSION['user_id']);
}

function is_admin() {
    return is_logged_in() && $_SESSION['user_type'] === 'admin';
}

function is_gestor() {
    return is_logged_in() && ($_SESSION['user_type'] === 'gestor' || $_SESSION['user_type'] === 'admin');
}

function require_login() {
    if (!is_logged_in()) {
        redirect('/login.php');
    }
}

function require_admin() {
    require_login();
    if (!is_admin()) {
        redirect('/dashboard.php?error=permission');
    }
}

function require_gestor() {
    require_login();
    if (!is_gestor()) {
        redirect('/dashboard.php?error=permission');
    }
}

// Função para exibir mensagens ao usuário
function set_flash_message($type, $message) {
    $_SESSION['flash'] = [
        'type' => $type,
        'message' => $message
    ];
}

function get_flash_message() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

// Função para formatar datas
function format_date($date, $format = 'd/m/Y H:i') {
    return date($format, strtotime($date));
}

// Função para formatar tamanho de arquivos
function format_file_size($size) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;
    while ($size >= 1024 && $i < count($units) - 1) {
        $size /= 1024;
        $i++;
    }
    return round($size, 2) . ' ' . $units[$i];
} 